<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://logrise.com
 * @since      0.1.0
 *
 * @package    Wp_Logrise_Utility
 * @subpackage Wp_Logrise_Utility/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      0.1.0
 * @package    Wp_Logrise_Utility
 * @subpackage Wp_Logrise_Utility/includes
 * @author     Logrise Lda <support@logrise.com>
 */
class Wp_Logrise_Utility
{

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    0.1.0
     * @access   protected
     * @var      Wp_Logrise_Utility_Loader $loader Maintains and registers all hooks for the plugin.
     */
    protected Wp_Logrise_Utility_Loader $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    0.1.0
     * @access   protected
     * @var      string $plugin_name The string used to uniquely identify this plugin.
     */
    protected string $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    0.1.0
     * @access   protected
     * @var      string $version The current version of the plugin.
     */
    protected string $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    0.1.0
     */
    public function __construct()
    {
        if (defined('WP_LOGRISE_UTILITY_VERSION')) {
            $this->version = WP_LOGRISE_UTILITY_VERSION;
        } else {
            $this->version = '0.1.0';
        }
        $this->plugin_name = 'wp-logrise-utility';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - Wp_Logrise_Utility_Loader. Orchestrates the hooks of the plugin.
     * - Wp_Logrise_Utility_i18n. Defines internationalization functionality.
     * - Wp_Logrise_Utility_Admin. Defines all hooks for the admin area.
     * - Wp_Logrise_Utility_Public. Defines all hooks for the public side of the site.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    0.1.0
     * @access   private
     */
    private function load_dependencies(): void
    {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-wp-logrise-utility-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-wp-logrise-utility-i18n.php';

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-wp-logrise-utility-admin.php';

        /**
         * The class responsible for defining all actions that occur in the public-facing
         * side of the site.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'public/class-wp-logrise-utility-public.php';

        /**
         * The class responsible for managing cache in the admin area.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-wp-logrise-utility-cache-manager.php';

        /**
         * The class responsible for handling plugin updates.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-wp-logrise-utility-updater.php';

        $this->loader = new Wp_Logrise_Utility_Loader();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Wp_Logrise_Utility_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    0.1.0
     * @access   private
     */
    private function set_locale(): void
    {

        $plugin_i18n = new Wp_Logrise_Utility_i18n();

        $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    0.1.0
     * @access   private
     */
    private function define_admin_hooks(): void
    {

        $plugin_admin = new Wp_Logrise_Utility_Admin($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');

        // Initialize cache manager
        $cache_manager = new Wp_Logrise_Utility_Cache_Manager(
            $this->get_plugin_name(),
            $this->get_version()
        );
        $cache_manager->init();

        // Initialize updater
        $updater = new Wp_Logrise_Utility_Updater();
        $updater->init();
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    0.1.0
     * @access   private
     */
    private function define_public_hooks(): void
    {

        $plugin_public = new Wp_Logrise_Utility_Public($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_styles');
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_scripts');
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    0.1.0
     */
    public function run(): void
    {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @return    string    The name of the plugin.
     * @since     0.1.0
     */
    public function get_plugin_name(): string
    {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @return    Wp_Logrise_Utility_Loader    Orchestrates the hooks of the plugin.
     * @since     0.1.0
     */
    public function get_loader(): Wp_Logrise_Utility_Loader
    {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @return    string    The version number of the plugin.
     * @since     0.1.0
     */
    public function get_version(): string
    {
        return $this->version;
    }
}
