<?php
/**
 * Class Wp_Logrise_Utility_Updater
 * Handles plugin updates from a custom update server
 *
 * @since      0.1.0
 * @package    Wp_Logrise_Utility
 * @subpackage Wp_Logrise_Utility/includes
 */
class Wp_Logrise_Utility_Updater
{

    /**
     * Plugin slug
     *
     * @var string
     */
    public $plugin_slug;

    /**
     * Plugin basename
     *
     * @var string
     */
    public $plugin_basename_file;

    /**
     * Plugin version
     *
     * @var string
     */
    public $version;

    /**
     * Cache key for updates
     *
     * @var string
     */
    public $cache_key;

    /**
     * Whether to allow caching
     *
     * @var bool
     */
    public $cache_allowed;

    /**
     * Initialize the updater
     */
    public function __construct()
    {
        $this->plugin_slug = plugin_basename(dirname(dirname(__FILE__)));
        $this->plugin_basename_file = plugin_basename(dirname(dirname(__FILE__))) . '/wp-logrise-utility.php';
        $this->version = defined('WP_LOGRISE_UTILITY_VERSION') ? WP_LOGRISE_UTILITY_VERSION : '0.1.0';
        $this->cache_key = 'wp_logrise_utility_update_check';
        $this->cache_allowed = true;
    }

    public function init()
    {
        add_filter('plugins_api', array($this, 'info'), 20, 3);
        add_filter('site_transient_update_plugins', array($this, 'update'));
        add_action('upgrader_process_complete', array($this, 'purge'), 10, 2);
    }

    /**
     * Make request to update server
     *
     * @return false|object
     */
    public function request()
    {
        $remote = get_transient($this->cache_key);

        if (isset($_GET['force-check']) && $_GET['force-check'] == 1) {
            $remote = false;
        }

        if (false === $remote || !$this->cache_allowed) {
            $remote = wp_remote_get(
                'https://updates.logrise.pt/info/wp-logrise-utility',
                array(
                    'timeout' => 10,
                    'headers' => array(
                        'Accept' => 'application/json'
                    )
                )
            );

            if (
                is_wp_error($remote)
                || 200 !== wp_remote_retrieve_response_code($remote)
                || empty(wp_remote_retrieve_body($remote))
            ) {
                return false;
            }

            set_transient($this->cache_key, $remote, DAY_IN_SECONDS);
        }

        return json_decode(wp_remote_retrieve_body($remote));
    }

    /**
     * Provide plugin information for the WordPress updates system
     */
    public function info($res, $action, $args)
    {
        if ('plugin_information' !== $action) {
            return $res;
        }

        if ($this->plugin_slug !== $args->slug) {
            return $res;
        }

        $remote = $this->request();

        if (!$remote) {
            return $res;
        }

        $res = new stdClass();

        $res->name = 'WP Logrise Utility';
        $res->slug = $remote->slug;
        $res->version = $remote->version;
        $res->tested = $remote->tested;
        $res->requires = $remote->requires;
        $res->author = 'Logrise Lda';
        $res->author_profile = $remote->author_profile;
        $res->download_link = $remote->download_url;
        $res->trunk = $remote->download_url;
        $res->requires_php = $remote->requires_php;
        $res->last_updated = $remote->last_updated;
        $res->sections = (array) $remote->sections;

        if (!empty($remote->banners)) {
            $res->banners = (array) $remote->banners;
        }

        return $res;
    }

    /**
     * Check for plugin updates
     */
    public function update($transient)
    {
        if (empty($transient->checked)) {
            return $transient;
        }

        $remote = $this->request();

        if (
            $remote
            && version_compare($this->version, $remote->version, '<')
            && version_compare($remote->requires, get_bloginfo('version'), '<=')
            && version_compare($remote->requires_php, PHP_VERSION, '<=')
        ) {
            $res = new stdClass();
            $res->slug = $this->plugin_slug;
            $res->plugin = $this->plugin_basename_file;
            $res->new_version = $remote->version;
            $res->tested = $remote->tested;
            $res->package = $remote->download_url;
            $res->icons = isset($remote->icons) ? (array) $remote->icons : array();
            $res->banners = isset($remote->banners) ? (array) $remote->banners : array();
            $res->banners_rtl = isset($remote->banners_rtl) ? (array) $remote->banners_rtl : array();
            $res->compatibility = new stdClass();

            $transient->response[$res->plugin] = $res;
        } else {
            $res = new stdClass();
            $res->slug = $this->plugin_slug;
            $res->plugin = $this->plugin_basename_file;
            $res->new_version = $this->version;
            $res->tested = '';
            $res->package = '';
            $res->icons = array();
            $res->banners = array();
            $res->banners_rtl = array();
            $res->compatibility = new stdClass();

            $transient->no_update[$res->plugin] = $res;
        }

        return $transient;
    }

    /**
     * Purge cache after update
     */
    public function purge($upgrader, $options)
    {
        if (
            $this->cache_allowed
            && 'update' === $options['action']
            && 'plugin' === $options['type']
        ) {
            delete_transient($this->cache_key);
        }
    }
}
